<?php
header('Content-Type: application/json');
require 'db.php';

date_default_timezone_set('UTC');

// 1. Check if the request method is OPTIONS (the preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    // 2. Respond with the necessary CORS headers for the preflight
    header("Access-Control-Allow-Origin: *"); // Allow your origin
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS"); // Allow the methods you use
    header("Access-Control-Allow-Headers: Content-Type, Authorization"); // Allow the custom headers you send
    header("HTTP/1.1 200 OK"); // **Crucially, ensure a 200 OK status**
    exit(); // Stop execution after sending the preflight response
}

// 3. Add the CORS header for the actual request (GET/POST) as well
header("Access-Control-Allow-Origin: *");

function respond(int $status, array $payload): void
{
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
    respond(405, ['success' => false, 'error' => 'Método no permitido']);
}

try {
    $data = json_decode(file_get_contents('php://input'), true, 512, JSON_THROW_ON_ERROR);
} catch (Throwable $error) {
    respond(400, ['success' => false, 'error' => 'JSON inválido']);
}

$productId = isset($data['id']) ? (int) $data['id'] : 0;
if ($productId <= 0) {
    respond(400, ['success' => false, 'error' => 'ID de producto inválido']);
}

try {
    $pdo->beginTransaction();

    $pdo->prepare('DELETE FROM product_toppings WHERE product_id = ?')->execute([$productId]);
    $pdo->prepare('DELETE FROM product_sizes WHERE product_id = ?')->execute([$productId]);
    $pdo->prepare('DELETE FROM reviews WHERE product_id = ?')->execute([$productId]);

    $deleteStmt = $pdo->prepare('DELETE FROM products WHERE id = ?');
    $deleteStmt->execute([$productId]);

    if ($deleteStmt->rowCount() === 0) {
        $pdo->rollBack();
        respond(404, ['success' => false, 'error' => 'Producto no encontrado']);
    }

    $pdo->commit();

    respond(200, ['success' => true]);
} catch (Throwable $error) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    respond(500, ['success' => false, 'error' => 'No se pudo eliminar el producto', 'detail' => $error->getMessage()]);
}
