<?php
header('Content-Type: application/json; charset=utf-8');
require_once 'db.php'; // Debe exponer $pdo (PDO conectado)

// CORS para peticiones directas y preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    header("Access-Control-Allow-Origin: *");
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
    header("Access-Control-Allow-Headers: Content-Type, Authorization");
    header("HTTP/1.1 200 OK");
    exit();
}
header("Access-Control-Allow-Origin: *");

function json_fail(string $msg): void {
    echo json_encode(['success' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}
function json_success(array $payload = []): void {
    echo json_encode(['success' => true] + $payload, JSON_UNESCAPED_UNICODE);
    exit;
}

$order_id = isset($_POST['order_id']) ? (int)$_POST['order_id'] : 0;
if ($order_id <= 0) json_fail('order_id requerido');

try {
    // Consulta principal con JOIN para el driver
    $stmt = $pdo->prepare(
        'SELECT o.*, 
                du.id AS driver_id, du.name AS driver_name, du.avatar AS driver_avatar, du.phone AS driver_phone, du.rating AS driver_rating
           FROM orders o
      LEFT JOIN users du ON du.id = o.driver_id
          WHERE o.id = ? LIMIT 1'
    );
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$order) json_fail('Orden no encontrada');

    // Consulta de items
    $itemsStmt = $pdo->prepare(
        'SELECT product_id, quantity, price, name, image
           FROM order_items
          WHERE order_id = ? ORDER BY id ASC'
    );
    $itemsStmt->execute([$order_id]);
    $items = [];
    while ($row = $itemsStmt->fetch(PDO::FETCH_ASSOC)) {
        $row['line_total'] = ((float)$row['quantity']) * ((float)$row['price']);
        $items[] = $row;
    }

    // Construir objeto driver
    $driver = null;
    if ($order['driver_id']) {
        $driver = [
            'id'     => (int)$order['driver_id'],
            'name'   => $order['driver_name'] ?? null,
            'avatar' => $order['driver_avatar'] ?? null,
            'phone'  => $order['driver_phone'] ?? null,
            'rating' => $order['driver_rating'] !== null ? (float)$order['driver_rating'] : null,
        ];
    }

    // Construir respuesta
    $output = [
        'id' => (int)$order['id'],
        'order_number' => $order['order_number'] ?? null,
        'user_id' => (int)$order['user_id'],
        'address_id' => (int)$order['address_id'],
        'total' => (float)$order['total'],
        'status' => $order['status'] ?? null,
        'notes' => $order['notes'] ?? null,
        'transfer_reference' => $order['transfer_reference'] ?? null,
        'receipt_filename' => $order['receipt_filename'] ?? null,
        'pickup_latitude' => $order['pickup_latitude'] !== null ? (float)$order['pickup_latitude'] : null,
        'pickup_longitude' => $order['pickup_longitude'] !== null ? (float)$order['pickup_longitude'] : null,
        'delivery_latitude' => $order['delivery_latitude'] !== null ? (float)$order['delivery_latitude'] : null,
        'delivery_longitude' => $order['delivery_longitude'] !== null ? (float)$order['delivery_longitude'] : null,
        'distance_km' => $order['distance_km'] !== null ? (float)$order['distance_km'] : null,
        'estimated_delivery_at' => $order['estimated_delivery_at'] ?? null,
        'created_at' => $order['created_at'] ?? null,
        'updated_at' => $order['updated_at'] ?? null,
        'driver' => $driver,
        'items' => $items,
    ];

    json_success(['order' => $output]);
} catch (Exception $e) {
    json_fail('DB error');
}
?>