<?php
header('Content-Type: application/json; charset=utf-8');
require_once 'db.php'; // Debe exponer $pdo (PDO conectado)

// 1. Check if the request method is OPTIONS (the preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    // 2. Respond with the necessary CORS headers for the preflight
    header("Access-Control-Allow-Origin: *"); // Allow your origin
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS"); // Allow the methods you use
    header("Access-Control-Allow-Headers: Content-Type, Authorization"); // Allow the custom headers you send
    header("HTTP/1.1 200 OK"); // **Crucially, ensure a 200 OK status**
    exit(); // Stop execution after sending the preflight response
}

// 3. Add the CORS header for the actual request (GET/POST) as well
header("Access-Control-Allow-Origin: *");

function json_fail(string $msg): void {
    echo json_encode(['success' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}

function json_success(array $payload = []): void {
    echo json_encode(['success' => true] + $payload, JSON_UNESCAPED_UNICODE);
    exit;
}

$rawBody = file_get_contents('php://input');
$jsonPayload = [];
if ($rawBody !== false && $rawBody !== '') {
    $decoded = json_decode($rawBody, true);
    if (is_array($decoded)) {
        $jsonPayload = $decoded;
    }
}

$user_id = 0;
if (isset($jsonPayload['user_id'])) {
    $user_id = (int)$jsonPayload['user_id'];
} elseif (isset($_POST['user_id'])) {
    $user_id = (int)$_POST['user_id'];
}

$status_filter = null;
if (isset($jsonPayload['status'])) {
    $status_filter = trim((string)$jsonPayload['status']);
} elseif (isset($_POST['status'])) {
    $status_filter = trim((string)$_POST['status']);
}
if ($status_filter === '') {
    $status_filter = null;
}

$statusMap = [
    'pending' => 'Pending',
    'processing' => 'Processing',
    'shipped' => 'Shipped',
    'delivered' => 'Delivered',
    'canceled' => 'Canceled'
];

$statusColors = [
    'pending' => '#FFB02C',
    'processing' => '#2B39B9',
    'shipped' => '#2B39B9',
    'delivered' => '#04BB66',
    'canceled' => '#C8102E'
];

try {
    $whereClauses = [];
    $params = [];

    if ($user_id > 0) {
        $whereClauses[] = 'o.user_id = ?';
        $params[] = $user_id;
    }

    if ($status_filter !== null) {
        $whereClauses[] = 'o.status = ?';
        $params[] = $status_filter;
    }

    $whereSql = $whereClauses ? ('WHERE ' . implode(' AND ', $whereClauses)) : '';

    $stmt = $pdo->prepare(
        'SELECT o.id, o.order_number, o.user_id, o.address_id, o.total, o.status, o.notes,
                o.transfer_reference, o.receipt_filename,
                o.pickup_latitude, o.pickup_longitude,
                o.delivery_latitude, o.delivery_longitude,
                o.distance_km, o.estimated_delivery_at,
                o.driver_name, o.driver_phone, o.driver_avatar, o.driver_rating,
                o.created_at, o.updated_at, o.driver_id,
                u.name AS customer_name, u.email AS customer_email, u.phone AS customer_phone,
                d.status_label,
                du.name AS driver_name, du.avatar AS driver_avatar, du.phone AS driver_phone, du.rating AS driver_rating
           FROM orders o
      LEFT JOIN users u ON u.id = o.user_id
      LEFT JOIN users du ON du.id = o.driver_id
      LEFT JOIN order_status_definitions d ON d.status_code = o.status
          ' . $whereSql . '
          ORDER BY o.created_at DESC'
    );
    $stmt->execute($params);
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $orderIds = array_column($orders, 'id');
    $itemsByOrder = [];

    if ($orderIds) {
        $placeholders = implode(',', array_fill(0, count($orderIds), '?'));
        $itemsStmt = $pdo->prepare(
            'SELECT order_id, product_id, quantity, price, name, image
               FROM order_items
              WHERE order_id IN (' . $placeholders . ')
              ORDER BY id ASC'
        );
        $itemsStmt->execute($orderIds);
        $itemsRows = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($itemsRows as $row) {
            $orderId = (int)$row['order_id'];
            if (!isset($itemsByOrder[$orderId])) {
                $itemsByOrder[$orderId] = [];
            }
            $lineTotal = ((float)$row['quantity']) * ((float)$row['price']);
            $itemsByOrder[$orderId][] = [
                'product_id' => (int)$row['product_id'],
                'quantity'   => (int)$row['quantity'],
                'price'      => (float)$row['price'],
                'name'       => $row['name'] !== null ? (string)$row['name'] : null,
                'image'      => $row['image'] !== null ? (string)$row['image'] : null,
                'line_total' => $lineTotal,
            ];
        }
    }

    $output = [];
    foreach ($orders as $order) {
        $id = (int)$order['id'];
        $items = $itemsByOrder[$id] ?? [];
        $itemCount = 0;
        foreach ($items as $item) {
            $itemCount += $item['quantity'];
        }
        $statusRaw = $order['status'];
        $statusCode = $statusRaw !== null ? strtolower((string)$statusRaw) : null;
        $statusLabel = null;
        $statusColor = null;

        if ($statusCode !== null) {
            if (isset($statusMap[$statusCode])) {
                $statusLabel = $statusMap[$statusCode];
            }
            if (isset($statusColors[$statusCode])) {
                $statusColor = $statusColors[$statusCode];
            }
        }

        if ($statusLabel === null && $order['status_label'] !== null) {
            $statusLabel = (string)$order['status_label'];
        }

        if ($statusLabel === null && $statusCode !== null) {
            $statusLabel = $statusCode;
        }

        $output[] = [
            'id'                   => $id,
            'order_number'         => $order['order_number'] !== null ? (string)$order['order_number'] : null,
            'user_id'              => (int)$order['user_id'],
            'address_id'           => (int)$order['address_id'],
            'total'                => (float)$order['total'],
            'status'               => $statusCode,
            'status_label'         => $statusLabel,
            'status_color'         => $statusColor,
            'notes'                => $order['notes'] !== null ? (string)$order['notes'] : null,
            'transfer_reference'   => $order['transfer_reference'] !== null ? (string)$order['transfer_reference'] : null,
            'receipt_filename'     => $order['receipt_filename'] !== null ? (string)$order['receipt_filename'] : null,
            'item_count'           => $itemCount,
            'pickup_latitude'      => $order['pickup_latitude']      !== null ? (float)$order['pickup_latitude'] : null,
            'pickup_longitude'     => $order['pickup_longitude']     !== null ? (float)$order['pickup_longitude'] : null,
            'delivery_latitude'    => $order['delivery_latitude']    !== null ? (float)$order['delivery_latitude'] : null,
            'delivery_longitude'   => $order['delivery_longitude']   !== null ? (float)$order['delivery_longitude'] : null,
            'distance_km'          => $order['distance_km']          !== null ? (float)$order['distance_km'] : null,
            'estimated_delivery_at'=> $order['estimated_delivery_at']!== null ? (string)$order['estimated_delivery_at'] : null,
            'driver_name'          => $order['driver_name']          !== null ? (string)$order['driver_name'] : null,
            'driver_phone'         => $order['driver_phone']         !== null ? (string)$order['driver_phone'] : null,
            'driver_avatar'        => $order['driver_avatar']        !== null ? (string)$order['driver_avatar'] : null,
            'driver_rating'        => $order['driver_rating']        !== null ? (float)$order['driver_rating'] : null,
            'created_at'           => $order['created_at'] !== null ? (string)$order['created_at'] : null,
            'updated_at'           => $order['updated_at'] !== null ? (string)$order['updated_at'] : null,
            // In $output[] mapping:
            'driver' => [
                'id'     => $order['driver_id'] !== null ? (int)$order['driver_id'] : null,
                'name'   => $order['driver_name'] !== null ? (string)$order['driver_name'] : null,
                'avatar' => $order['driver_avatar'] !== null ? (string)$order['driver_avatar'] : null,
                'phone'  => $order['driver_phone'] !== null ? (string)$order['driver_phone'] : null,
                'rating' => $order['driver_rating'] !== null ? (float)$order['driver_rating'] : null,
            ],
            'customer'             => [
                'name'  => $order['customer_name'] !== null ? (string)$order['customer_name'] : null,
                'email' => $order['customer_email'] !== null ? (string)$order['customer_email'] : null,
                'phone' => $order['customer_phone'] !== null ? (string)$order['customer_phone'] : null,
            ],
            'items'                => $items,
        ];
    }

    json_success(['orders' => $output]);
} catch (Exception $e) {
    json_fail('DB error');
}