<?php
error_reporting(E_ALL);
ini_set('display_errors', 0);
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/db.php'; // <-- usa tu conexión

define('DEBUG', isset($_GET['debug']) && $_GET['debug'] == '1');
define('LOG_FILE', __DIR__ . '/save_debug.log');

function debug_log($label, $data) {
  if (!DEBUG) return;
  $line = date('c') . ' [' . $label . '] ' . (is_scalar($data) ? $data : json_encode($data, JSON_UNESCAPED_UNICODE)) . PHP_EOL;
  file_put_contents(LOG_FILE, $line, FILE_APPEND);
}

ob_start();

function fail($code, $msg, $status = 400, $extra = []) {
  http_response_code($status);
  $payload = ['success'=>false,'error_code'=>$code,'error'=>$msg];
  if (DEBUG && $extra) $payload['debug'] = $extra;
  $buf = ob_get_clean();
  if (DEBUG && $buf) $payload['debug_output'] = $buf;
  echo json_encode($payload);
  exit;
}

$raw = file_get_contents('php://input');
debug_log('RAW', $raw);

$data = [];
$contentType = $_SERVER['CONTENT_TYPE'] ?? '';
$isJson = stripos($contentType, 'application/json') !== false;

if ($isJson) {
  $data = json_decode($raw, true);
  if ($data === null && json_last_error() !== JSON_ERROR_NONE) {
    fail('json_parse_error','Invalid JSON payload',400,['json_error'=>json_last_error_msg()]);
  }
} else {
  $data = $_POST;
  if (isset($_POST['items'])) {
    $decoded = json_decode($_POST['items'], true);
    if ($decoded === null && json_last_error() !== JSON_ERROR_NONE) {
      fail('json_items_parse_error','Invalid JSON in items',400,['json_error'=>json_last_error_msg()]);
    }
    $data['items'] = $decoded;
  }
}

debug_log('PARSED', $data);

$required = ['user_id','items','total','address_id'];
foreach ($required as $f) {
  if (!isset($data[$f])) fail('missing_field',"Missing field: $f");
}

$user_id = (int)$data['user_id'];
$address_id = (int)$data['address_id'];
$total = (float)$data['total'];
$items = $data['items'];

if (!is_array($items) || count($items) === 0) fail('invalid_items','Items must be a non-empty array');

$calcTotal = 0.0;
foreach ($items as $idx => $it) {
  if (!isset($it['product_id'],$it['quantity'],$it['price'])) {
    fail('invalid_item','Malformed item',400,['index'=>$idx,'item'=>$it]);
  }
  $pid = (int)$it['product_id'];
  $qty = (int)$it['quantity'];
  $price = (float)$it['price'];
  if ($pid <= 0 || $qty <= 0 || $price < 0) {
    fail('invalid_item_values','Invalid product_id/quantity/price',400,['index'=>$idx]);
  }
  $calcTotal += $qty * $price;
}

if (abs($calcTotal - $total) > 0.02) {
  fail('total_mismatch','Provided total does not match items sum',400,['calc'=>$calcTotal,'provided'=>$total]);
}

$receipt_filename = isset($data['receipt_filename']) ? basename($data['receipt_filename']) : null;
if ($receipt_filename) {
  $expected = __DIR__.'/uploads/receipts/'.$receipt_filename;
  if (!file_exists($expected)) {
    fail('receipt_missing','Receipt file not found',400,['path'=>$expected]);
  }
}

$notes = isset($data['notes']) ? trim($data['notes']) : null;
$transfer_ref = isset($data['transfer_reference']) ? trim($data['transfer_reference']) : null;
$status = isset($data['status']) ? substr($data['status'],0,30) : 'pending';

// Conexión desde db.php
if (!isset($pdo)) {
  fail('db_connect_error','DB connection not available');
}

// Validar address
try {
  $st = $pdo->prepare('SELECT id FROM addresses WHERE id = ? AND user_id = ? LIMIT 1');
  $st->execute([$address_id, $user_id]);
  if (!$st->fetch()) fail('address_not_found','Address does not belong to user');
} catch (PDOException $e) {
  fail('db_query_error','Error validating address',500,['pdo'=>$e->getMessage()]);
}

$pdo->beginTransaction();
try {
  $datePart = date('ymd');
  $c = $pdo->query('SELECT COUNT(*) c FROM orders WHERE DATE(created_at)=CURRENT_DATE')->fetch(PDO::FETCH_ASSOC);
  $seq = (int)$c['c'] + 1;
  $orderNumber = sprintf('ORD-%s-%04d',$datePart,$seq);

  $ins = $pdo->prepare('INSERT INTO orders (user_id,address_id,total,status,notes,transfer_reference,receipt_filename,order_number,created_at) VALUES (?,?,?,?,?,?,?,?,NOW())');
  $ins->execute([$user_id,$address_id,$total,$status,$notes,$transfer_ref,$receipt_filename,$orderNumber]);
  $orderId = (int)$pdo->lastInsertId();

  $itemIns = $pdo->prepare('INSERT INTO order_items (order_id,product_id,quantity,price) VALUES (?,?,?,?)');
  foreach ($items as $it) {
    $itemIns->execute([$orderId,(int)$it['product_id'],(int)$it['quantity'],(float)$it['price']]);
  }

  $pdo->commit();
  $out = [
    'success'=>true,
    'order_id'=>$orderId,
    'order_number'=>$orderNumber,
    'message'=>'Order created'
  ];
  if (DEBUG) {
    $out['debug'] = [
      'calc_total'=>$calcTotal,
      'item_count'=>count($items),
      'receipt_filename'=>$receipt_filename,
      'status_used'=>$status
    ];
  }
  $buf = ob_get_clean();
  if (DEBUG && $buf) $out['debug_output'] = $buf;
  echo json_encode($out);
} catch (Exception $e) {
  $pdo->rollBack();
  fail('db_error','Database error: '.$e->getMessage(),500);
}