<?php
header('Content-Type: application/json; charset=utf-8');
require_once 'db.php'; // Debe exponer $pdo (PDO conectado)

// Handle CORS preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    header('Access-Control-Allow-Origin: *');
    header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
    header('Access-Control-Allow-Headers: Content-Type, Authorization');
    http_response_code(200);
    exit;
}

header('Access-Control-Allow-Origin: *');

function json_fail($msg) {
    echo json_encode(['success' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}

function json_success(array $payload = []) {
    echo json_encode(['success' => true] + $payload, JSON_UNESCAPED_UNICODE);
    exit;
}

// Support JSON payloads
$rawBody = file_get_contents('php://input');
$jsonPayload = [];
if ($rawBody !== false && $rawBody !== '') {
    $decoded = json_decode($rawBody, true);
    if (is_array($decoded)) {
        $jsonPayload = $decoded;
    }
}

// Helper to read field from JSON first, then POST fallback
$readField = static function (string $key) use ($jsonPayload) {
    if (array_key_exists($key, $jsonPayload)) {
        return $jsonPayload[$key];
    }
    if (array_key_exists($key, $_POST)) {
        return $_POST[$key];
    }
    return null;
};

$order_idRaw = $readField('order_id');
$status_raw = $readField('status');
$user_idRaw = $readField('user_id');
$noteRaw = $readField('note');

$order_id    = $order_idRaw !== null ? (int)$order_idRaw : 0;
$status_code = $status_raw !== null ? trim((string)$status_raw) : '';
if (!$order_id || $status_code === '') {
    json_fail('Missing order_id or status');
}

$user_id = $user_idRaw !== null ? (int)$user_idRaw : 0; // opcional para validar due09o
$note    = $noteRaw !== null ? trim((string)$noteRaw) : null;

$metaKeys = [
    'estimated_delivery_at',
    'pickup_latitude',
    'pickup_longitude',
    'delivery_latitude',
    'delivery_longitude',
    'distance_km',
    'driver_name',
    'driver_phone',
    'driver_avatar',
    'driver_rating',
];

$meta = [];
foreach ($metaKeys as $key) {
    $value = $readField($key);
    if ($value !== null && $value !== '') {
        $meta[$key] = $value;
    }
}

try {
    $pdo->beginTransaction();

    $stmt = $pdo->prepare('SELECT user_id FROM orders WHERE id = ? FOR UPDATE');
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        $pdo->rollBack();
        json_fail('Order not found');
    }

    if ($user_id && (int)$order['user_id'] !== $user_id) {
        $pdo->rollBack();
        json_fail('Order not owned by user');
    }

    $setParts = ['status = ?'];
    $params   = [$status_code];

    $numericFields = ['pickup_latitude', 'pickup_longitude', 'delivery_latitude', 'delivery_longitude', 'distance_km', 'driver_rating'];
    foreach ($meta as $key => $value) {
        $setParts[] = "{$key} = ?";
        $params[] = in_array($key, $numericFields, true) ? (float)$value : (string)$value;
    }

    $params[] = $order_id;
    $sql = 'UPDATE orders SET ' . implode(', ', $setParts) . ', updated_at = NOW() WHERE id = ?';
    $updateStmt = $pdo->prepare($sql);
    $updateStmt->execute($params);

    /*$labelStmt = $pdo->prepare('SELECT status_label FROM order_status_definitions WHERE status_code = ? LIMIT 1');
    $labelStmt->execute([$status_code]);
    $labelRow = $labelStmt->fetch(PDO::FETCH_ASSOC);
    $status_label = $labelRow['status_label'] ?? $status_code;*/

    $historyStmt = $pdo->prepare(
        'INSERT INTO order_status_history (order_id, status, note)
         VALUES (?, ?, ?)'
    );
    $historyStmt->execute([$order_id, $status_code, $note]);

    $pdo->commit();

    json_success([
        'order_id'     => $order_id,
        'status'       => $status_code,
        'status_label' => $status_label,
        'note'         => $note,
    ]);
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    json_fail('Failed to update order status');
}