<?php
header('Content-Type: application/json');
require 'db.php';

date_default_timezone_set('UTC');

// 1. Check if the request method is OPTIONS (the preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    // 2. Respond with the necessary CORS headers for the preflight
    header("Access-Control-Allow-Origin: *"); // Allow your origin
    header("Access-Control-Allow-Methods: GET, POST, OPTIONS"); // Allow the methods you use
    header("Access-Control-Allow-Headers: Content-Type, Authorization"); // Allow the custom headers you send
    header("HTTP/1.1 200 OK"); // **Crucially, ensure a 200 OK status**
    exit(); // Stop execution after sending the preflight response
}

// 3. Add the CORS header for the actual request (GET/POST) as well
header("Access-Control-Allow-Origin: *");

function respond(int $status, array $payload): void
{
    http_response_code($status);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
    respond(405, ['success' => false, 'error' => 'Método no permitido']);
}

try {
    $data = json_decode(file_get_contents('php://input'), true, 512, JSON_THROW_ON_ERROR);
} catch (Throwable $error) {
    respond(400, ['success' => false, 'error' => 'JSON inválido']);
}

$productId = isset($data['id']) ? (int) $data['id'] : 0;
if ($productId <= 0) {
    respond(400, ['success' => false, 'error' => 'ID de producto inválido']);
}

$stmt = $pdo->prepare('SELECT * FROM products WHERE id = ? LIMIT 1');
$stmt->execute([$productId]);
$current = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$current) {
    respond(404, ['success' => false, 'error' => 'Producto no encontrado']);
}

$title = array_key_exists('title', $data) ? trim((string) $data['title']) : $current['title'];
$categoryId = array_key_exists('category_id', $data) ? (int) $data['category_id'] : (int) $current['category_id'];
$description = array_key_exists('description', $data) ? (trim((string) $data['description']) ?: null) : ($current['description'] ?: null);
$image = array_key_exists('image', $data) ? (trim((string) $data['image']) ?: null) : ($current['image'] ?: null);
$type = array_key_exists('type', $data) ? (trim((string) $data['type']) ?: null) : ($current['type'] ?: null);
$markRaw = $data['mark'] ?? $data['featured'] ?? $current['mark'];
$mark = filter_var($markRaw, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
if ($mark === null) {
    $markValue = (int) $current['mark'];
} else {
    $markValue = $mark ? 1 : 0;
}

if ($title === '') {
    respond(400, ['success' => false, 'error' => 'El título no puede estar vacío']);
}

if ($categoryId <= 0) {
    respond(400, ['success' => false, 'error' => 'Categoría inválida']);
}

$shouldUpdateSizes = array_key_exists('sizes', $data);
$shouldUpdateToppings = array_key_exists('toppings', $data);

$normalizedSizes = [];
if ($shouldUpdateSizes) {
    $sizesPayload = $data['sizes'];
    if (!is_array($sizesPayload) || count($sizesPayload) === 0) {
        respond(400, ['success' => false, 'error' => 'Debes proporcionar al menos un tamaño']);
    }

    foreach ($sizesPayload as $size) {
        if (!is_array($size)) {
            continue;
        }
        $name = trim((string) ($size['name'] ?? ''));
        $price = $size['price'] ?? null;
        if ($name === '' || $price === null || $price === '') {
            continue;
        }
        $priceValue = is_numeric($price) ? (float) $price : null;
        if ($priceValue === null) {
            continue;
        }
        $slicesRaw = $size['slices'] ?? null;
        $slices = is_numeric($slicesRaw) ? (int) $slicesRaw : null;

        $normalizedSizes[] = [
            'name' => $name,
            'slices' => $slices,
            'price' => $priceValue
        ];
    }

    if (count($normalizedSizes) === 0) {
        respond(400, ['success' => false, 'error' => 'Los tamaños proporcionados no son válidos']);
    }
}

$normalizedToppings = [];
if ($shouldUpdateToppings) {
    $toppingsPayload = $data['toppings'];
    if (!is_array($toppingsPayload)) {
        $toppingsPayload = [];
    }

    foreach ($toppingsPayload as $topping) {
        if (!is_array($topping)) {
            continue;
        }
        $name = trim((string) ($topping['name'] ?? ''));
        $price = $topping['price'] ?? null;
        if ($name === '' || $price === null || $price === '') {
            continue;
        }
        $priceValue = is_numeric($price) ? (float) $price : null;
        if ($priceValue === null) {
            continue;
        }

        $normalizedToppings[] = [
            'name' => $name,
            'price' => $priceValue
        ];
    }
}

try {
    $pdo->beginTransaction();

    $updateStmt = $pdo->prepare('UPDATE products SET category_id = ?, title = ?, description = ?, image = ?, type = ?, mark = ? WHERE id = ?');
    $updateStmt->execute([$categoryId, $title, $description, $image, $type, $markValue, $productId]);

    if ($shouldUpdateSizes) {
        $pdo->prepare('DELETE FROM product_sizes WHERE product_id = ?')->execute([$productId]);
        $sizeStmt = $pdo->prepare('INSERT INTO product_sizes (product_id, name, slices, price) VALUES (?, ?, ?, ?)');
        foreach ($normalizedSizes as $size) {
            $sizeStmt->execute([$productId, $size['name'], $size['slices'], $size['price']]);
        }
    }

    if ($shouldUpdateToppings) {
        $pdo->prepare('DELETE FROM product_toppings WHERE product_id = ?')->execute([$productId]);
        if (count($normalizedToppings) > 0) {
            $toppingStmt = $pdo->prepare('INSERT INTO product_toppings (product_id, name, price) VALUES (?, ?, ?)');
            foreach ($normalizedToppings as $topping) {
                $toppingStmt->execute([$productId, $topping['name'], $topping['price']]);
            }
        }
    }

    $pdo->commit();

    respond(200, ['success' => true, 'product_id' => $productId]);
} catch (Throwable $error) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    respond(500, ['success' => false, 'error' => 'No se pudo actualizar el producto', 'detail' => $error->getMessage()]);
}
