<?php
header('Content-Type: application/json');
require 'db.php'; // Debe exponer $pdo (PDO)

function respond(bool $success, array $payload = [], int $statusCode = 200): void {
    http_response_code($statusCode);
    echo json_encode(['success' => $success] + $payload);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    respond(false, ['error' => 'Método no permitido'], 405);
}

function loadPayload(): array {
    $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
    if (stripos($contentType, 'application/json') !== false) {
        $raw = file_get_contents('php://input');
        $decoded = json_decode($raw, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            respond(false, ['error' => 'JSON inválido: ' . json_last_error_msg()], 400);
        }
        return $decoded ?? [];
    }
    return $_POST;
}

$data = loadPayload();

$addressLine1 = trim((string) ($data['address_line1'] ?? $data['address'] ?? ''));
$postalCode   = trim((string) ($data['postal_code'] ?? $data['pincode'] ?? ''));
$city         = trim((string) ($data['city'] ?? ''));
$state        = trim((string) ($data['state'] ?? ''));
$country      = trim((string) ($data['country'] ?? ''));
if ($city === '' && $postalCode !== '') {
    $city = null; // opcional: podrías mapear según tu lógica
}
if ($state === '') {
    $state = null;
}
if ($country === '') {
    $country = null;
}

$required = [
    'user_id'       => isset($data['user_id']) ? (int) $data['user_id'] : null,
    'name'          => trim((string) ($data['name'] ?? '')),
    'phone'         => trim((string) ($data['phone'] ?? '')),
    'address_line1' => $addressLine1,
    'postal_code'   => $postalCode,
];

$missing = array_keys(array_filter($required, fn($value) => $value === null || $value === ''));

if (!empty($missing)) {
    respond(false, [
        'error'  => 'Faltan campos obligatorios',
        'fields' => $missing,
    ], 422);
}

$userId    = (int) $required['user_id'];
$name      = $required['name'];
$phone     = $required['phone'];
$line2     = isset($data['address_line2']) ? trim((string) $data['address_line2']) : null;
$isDefault = filter_var($data['is_default'] ?? false, FILTER_VALIDATE_BOOLEAN);
$latitude  = isset($data['latitude']) && $data['latitude'] !== '' ? (float) $data['latitude'] : null;
$longitude = isset($data['longitude']) && $data['longitude'] !== '' ? (float) $data['longitude'] : null;
$addressId = isset($data['address_id']) && $data['address_id'] !== '' ? (int) $data['address_id'] : null;

try {
    $pdo->beginTransaction();

    if ($isDefault) {
        $reset = $pdo->prepare('UPDATE addresses SET is_default = 0 WHERE user_id = ?');
        $reset->execute([$userId]);
    }

    if ($addressId) {
        $update = $pdo->prepare(
            'UPDATE addresses SET
                name = :name,
                phone = :phone,
                address_line1 = :line1,
                address_line2 = :line2,
                city = :city,
                state = :state,
                country = :country,
                postal_code = :postal_code,
                latitude = :latitude,
                longitude = :longitude,
                is_default = :is_default,
                updated_at = NOW()
             WHERE id = :id AND user_id = :user_id'
        );
        $update->execute([
            ':name'        => $name,
            ':phone'       => $phone,
            ':line1'       => $addressLine1,
            ':line2'       => $line2,
            ':city'        => $city,
            ':state'       => $state,
            ':country'     => $country,
            ':postal_code' => $postalCode,
            ':latitude'    => $latitude,
            ':longitude'   => $longitude,
            ':is_default'  => $isDefault ? 1 : 0,
            ':id'          => $addressId,
            ':user_id'     => $userId,
        ]);

        if ($update->rowCount() === 0) {
            throw new RuntimeException('Dirección no encontrada o sin cambios.');
        }

        $savedId = $addressId;
    } else {
        $insert = $pdo->prepare(
            'INSERT INTO addresses (
                user_id, name, phone, address_line1, address_line2,
                city, state, country, postal_code, latitude, longitude,
                is_default, created_at, updated_at
            ) VALUES (
                :user_id, :name, :phone, :line1, :line2,
                :city, :state, :country, :postal_code, :latitude, :longitude,
                :is_default, NOW(), NOW()
            )'
        );
        $insert->execute([
            ':user_id'     => $userId,
            ':name'        => $name,
            ':phone'       => $phone,
            ':line1'       => $addressLine1,
            ':line2'       => $line2,
            ':city'        => $city,
            ':state'       => $state,
            ':country'     => $country,
            ':postal_code' => $postalCode,
            ':latitude'    => $latitude,
            ':longitude'   => $longitude,
            ':is_default'  => $isDefault ? 1 : 0,
        ]);

        $savedId = (int) $pdo->lastInsertId();
    }

    $pdo->commit();

    $fetch = $pdo->prepare('SELECT * FROM addresses WHERE id = ?');
    $fetch->execute([$savedId]);
    $address = $fetch->fetch(PDO::FETCH_ASSOC);

    respond(true, [
        'message' => $addressId ? 'Dirección actualizada correctamente' : 'Dirección guardada correctamente',
        'address' => $address,
    ]);
} catch (Throwable $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    respond(false, ['error' => $e->getMessage()], 500);
}