<?php
header('Content-Type: application/json; charset=utf-8');
require_once 'db.php'; // Debe exponer $pdo (PDO conectado)

function json_fail($msg) {
    echo json_encode(['success' => false, 'error' => $msg], JSON_UNESCAPED_UNICODE);
    exit;
}

function json_success(array $payload = []) {
    echo json_encode(['success' => true] + $payload, JSON_UNESCAPED_UNICODE);
    exit;
}

$order_id    = isset($_POST['order_id']) ? (int)$_POST['order_id'] : 0;
$status_code = isset($_POST['status']) ? trim((string)$_POST['status']) : '';
if (!$order_id || $status_code === '') {
    json_fail('Missing order_id or status');
}

$user_id = isset($_POST['user_id']) ? (int)$_POST['user_id'] : 0; // opcional para validar due09o
$note    = isset($_POST['note']) ? trim((string)$_POST['note']) : null;

$metaKeys = [
    'estimated_delivery_at',
    'pickup_latitude',
    'pickup_longitude',
    'delivery_latitude',
    'delivery_longitude',
    'distance_km',
    'driver_name',
    'driver_phone',
    'driver_avatar',
    'driver_rating',
];

$meta = [];
foreach ($metaKeys as $key) {
    if (array_key_exists($key, $_POST) && $_POST[$key] !== '') {
        $meta[$key] = $_POST[$key];
    }
}

try {
    $pdo->beginTransaction();

    $stmt = $pdo->prepare('SELECT user_id FROM orders WHERE id = ? FOR UPDATE');
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$order) {
        $pdo->rollBack();
        json_fail('Order not found');
    }

    if ($user_id && (int)$order['user_id'] !== $user_id) {
        $pdo->rollBack();
        json_fail('Order not owned by user');
    }

    $setParts = ['status = ?'];
    $params   = [$status_code];

    $numericFields = ['pickup_latitude', 'pickup_longitude', 'delivery_latitude', 'delivery_longitude', 'distance_km', 'driver_rating'];
    foreach ($meta as $key => $value) {
        $setParts[] = "{$key} = ?";
        $params[] = in_array($key, $numericFields, true) ? (float)$value : (string)$value;
    }

    $params[] = $order_id;
    $sql = 'UPDATE orders SET ' . implode(', ', $setParts) . ', updated_at = NOW() WHERE id = ?';
    $updateStmt = $pdo->prepare($sql);
    $updateStmt->execute($params);

    $labelStmt = $pdo->prepare('SELECT status_label FROM order_status_definitions WHERE status_code = ? LIMIT 1');
    $labelStmt->execute([$status_code]);
    $labelRow = $labelStmt->fetch(PDO::FETCH_ASSOC);
    $status_label = $labelRow['status_label'] ?? $status_code;

    $historyStmt = $pdo->prepare(
        'INSERT INTO order_status_history (order_id, status_code, status_label, note)
         VALUES (?, ?, ?, ?)'
    );
    $historyStmt->execute([$order_id, $status_code, $status_label, $note]);

    $pdo->commit();

    json_success([
        'order_id'     => $order_id,
        'status'       => $status_code,
        'status_label' => $status_label,
        'note'         => $note,
    ]);
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    json_fail('Failed to update order status');
}